<?php

namespace App\Filament\Resources;

use App\Filament\Resources\RoomResource\Pages;
use App\Models\Room;
use App\Models\RoomType;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;

class RoomResource extends Resource
{
    protected static ?string $model = Room::class;

    protected static ?string $navigationIcon = 'heroicon-o-home';

    protected static ?string $navigationGroup = 'Hotel Management';

    protected static ?int $navigationSort = 2;

    /**
     * Define the form schema for creating and editing rooms.
     */
    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make()
                    ->schema([
                        Forms\Components\TextInput::make('room_number')
                            ->required()
                            ->unique(ignoreRecord: true)
                            ->maxLength(255)
                            ->label('Room Number')
                            ->placeholder('e.g., 101, 102, 201'),

                        Forms\Components\Select::make('room_type_id')
                            ->required()
                            ->relationship('roomType', 'name')
                            ->searchable()
                            ->preload()
                            ->createOptionForm([
                                Forms\Components\TextInput::make('name')
                                    ->required()
                                    ->unique()
                                    ->maxLength(255),
                                Forms\Components\TextInput::make('base_price')
                                    ->required()
                                    ->numeric()
                                    ->prefix('KSh')
                                    ->minValue(0),
                            ])
                            ->label('Room Type'),

                        Forms\Components\Select::make('status')
                            ->required()
                            ->options([
                                Room::STATUS_AVAILABLE => 'Available',
                                Room::STATUS_RESERVED => 'Reserved',
                                Room::STATUS_OCCUPIED => 'Occupied',
                                Room::STATUS_CLEANING => 'Cleaning',
                                Room::STATUS_MAINTENANCE => 'Maintenance',
                            ])
                            ->default(Room::STATUS_AVAILABLE)
                            ->label('Status')
                            ->native(false),

                        Forms\Components\TextInput::make('floor')
                            ->numeric()
                            ->minValue(0)
                            ->label('Floor Number')
                            ->placeholder('e.g., 1, 2, 3'),
                    ])
                    ->columns(2),
            ]);
    }

    /**
     * Define the table schema for listing rooms.
     */
    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('id')
                    ->label('ID')
                    ->sortable(),

                Tables\Columns\TextColumn::make('room_number')
                    ->searchable()
                    ->sortable()
                    ->label('Room Number'),

                Tables\Columns\TextColumn::make('roomType.name')
                    ->searchable()
                    ->sortable()
                    ->label('Room Type'),

                Tables\Columns\BadgeColumn::make('status')
                    ->sortable()
                    ->label('Status')
                    ->colors([
                        'success' => Room::STATUS_AVAILABLE,
                        'warning' => Room::STATUS_RESERVED,
                        'danger' => Room::STATUS_OCCUPIED,
                        'secondary' => Room::STATUS_CLEANING,
                        'primary' => Room::STATUS_MAINTENANCE,
                    ])
                    ->formatStateUsing(fn (string $state): string => ucfirst($state)),

                Tables\Columns\TextColumn::make('floor')
                    ->sortable()
                    ->label('Floor')
                    ->toggleable(),

                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('status')
                    ->options([
                        Room::STATUS_AVAILABLE => 'Available',
                        Room::STATUS_RESERVED => 'Reserved',
                        Room::STATUS_OCCUPIED => 'Occupied',
                        Room::STATUS_CLEANING => 'Cleaning',
                        Room::STATUS_MAINTENANCE => 'Maintenance',
                    ])
                    ->label('Status'),

                Tables\Filters\SelectFilter::make('room_type_id')
                    ->relationship('roomType', 'name')
                    ->label('Room Type'),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListRooms::route('/'),
            'create' => Pages\CreateRoom::route('/create'),
            'edit' => Pages\EditRoom::route('/{record}/edit'),
        ];
    }
}
