<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Room;
use App\Models\RoomType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;
use Inertia\Response;

class RoomManagementController extends Controller
{
    public function index(): Response
    {
        return Inertia::render('Admin/Rooms', [
            'rooms' => Room::query()
                ->with('roomType:id,name')
                ->orderBy('room_number')
                ->get()
                ->map(fn (Room $room) => [
                    'id' => $room->id,
                    'room_number' => $room->room_number,
                    'floor' => $room->floor,
                    'status' => $room->status,
                    'room_type_id' => $room->room_type_id,
                    'room_type_name' => $room->roomType?->name,
                    'image_url' => $room->image_path ? Storage::url($room->image_path) : null,
                ]),
            'roomTypes' => RoomType::query()->select('id', 'name')->orderBy('name')->get(),
            'statuses' => [
                Room::STATUS_AVAILABLE,
                Room::STATUS_RESERVED,
                Room::STATUS_OCCUPIED,
                Room::STATUS_CLEANING,
                Room::STATUS_MAINTENANCE,
            ],
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'room_type_id' => ['required', 'exists:room_types,id'],
            'room_number' => ['required', 'string', 'max:20', 'unique:rooms,room_number'],
            'floor' => ['nullable', 'integer', 'min:0'],
            'status' => ['required', 'in:' . implode(',', [
                Room::STATUS_AVAILABLE,
                Room::STATUS_RESERVED,
                Room::STATUS_OCCUPIED,
                Room::STATUS_CLEANING,
                Room::STATUS_MAINTENANCE,
            ])],
            'image' => ['nullable', 'image', 'max:4096'],
        ]);

        $imagePath = $request->file('image')?->store('rooms', 'public');

        Room::create([
            ...$validated,
            'image_path' => $imagePath,
        ]);

        return back()->with('success', 'Room created successfully.');
    }

    public function update(Request $request, Room $room)
    {
        $validated = $request->validate([
            'room_type_id' => ['required', 'exists:room_types,id'],
            'room_number' => ['required', 'string', 'max:20', 'unique:rooms,room_number,' . $room->id],
            'floor' => ['nullable', 'integer', 'min:0'],
            'status' => ['required', 'in:' . implode(',', [
                Room::STATUS_AVAILABLE,
                Room::STATUS_RESERVED,
                Room::STATUS_OCCUPIED,
                Room::STATUS_CLEANING,
                Room::STATUS_MAINTENANCE,
            ])],
            'image' => ['nullable', 'image', 'max:4096'],
        ]);

        $imagePath = $room->image_path;

        if ($request->hasFile('image')) {
            if ($imagePath) {
                Storage::disk('public')->delete($imagePath);
            }

            $imagePath = $request->file('image')->store('rooms', 'public');
        }

        $room->update([
            ...$validated,
            'image_path' => $imagePath,
        ]);

        return back()->with('success', 'Room updated successfully.');
    }
}
