<?php

namespace App\Services;

class ImageOptimizationService
{
    /**
     * Optimize image for web use - compress and generate multiple formats
     * Uses PHP's built-in GD library for image manipulation
     * 
     * @param string $imagePath Full file path to the image
     * @param array $sizes Array of sizes to generate: ['medium' => 800, 'small' => 400]
     * @return array Array of generated image paths
     */
    public static function optimizeImage(string $imagePath, array $sizes = []): array
    {
        if (!file_exists($imagePath) || !extension_loaded('gd')) {
            \Log::warning('Image optimization skipped: File not found or GD extension missing');
            return [];
        }

        $results = [];
        $pathInfo = pathinfo($imagePath);
        $basePath = $pathInfo['dirname'];
        $filename = $pathInfo['filename'];
        $ext = strtolower($pathInfo['extension']);

        try {
            // Determine image type and create image resource
            switch ($ext) {
                case 'png':
                    $originalImage = imagecreatefrompng($imagePath);
                    break;
                case 'gif':
                    $originalImage = imagecreatefromgif($imagePath);
                    break;
                case 'webp':
                    if (function_exists('imagecreatefromwebp')) {
                        $originalImage = imagecreatefromwebp($imagePath);
                    } else {
                        \Log::warning('WebP support not available in GD');
                        return [];
                    }
                    break;
                case 'jpg':
                case 'jpeg':
                default:
                    $originalImage = imagecreatefromjpeg($imagePath);
            }

            if (!$originalImage) {
                \Log::error('Failed to create image resource from: ' . $imagePath);
                return [];
            }

            // Optimize and save original as JPEG
            imagejpeg($originalImage, $imagePath, 75);
            $results['original'] = $imagePath;

            // Generate WebP version if supported
            if (function_exists('imagewebp')) {
                $webpPath = "{$basePath}/{$filename}.webp";
                imagewebp($originalImage, $webpPath, 70);
                $results['webp'] = $webpPath;
            }

            // Generate responsive sizes
            $defaultSizes = $sizes ?: ['large' => 1200, 'medium' => 800, 'small' => 400];

            $originalWidth = imagesx($originalImage);
            $originalHeight = imagesy($originalImage);

            foreach ($defaultSizes as $name => $width) {
                // Skip if original is smaller than target width
                if ($originalWidth < $width) {
                    continue;
                }

                // Calculate new height maintaining aspect ratio
                $height = (int) ($originalHeight * ($width / $originalWidth));

                // Create new image resource
                $resized = imagecreatetruecolor($width, $height);

                // Preserve transparency for PNG
                imagealphablending($resized, false);
                imagesavealpha($resized, true);

                // Resize
                imagecopyresampled($resized, $originalImage, 0, 0, 0, 0, $width, $height, $originalWidth, $originalHeight);

                // Save JPEG version
                $jpegPath = "{$basePath}/{$filename}-{$name}.jpg";
                imagejpeg($resized, $jpegPath, 75);
                $results["{$name}_jpeg"] = $jpegPath;

                // Save WebP version if supported
                if (function_exists('imagewebp')) {
                    $webpResizedPath = "{$basePath}/{$filename}-{$name}.webp";
                    imagewebp($resized, $webpResizedPath, 70);
                    $results["{$name}_webp"] = $webpResizedPath;
                }

                imagedestroy($resized);
            }

            imagedestroy($originalImage);

            return $results;
        } catch (\Exception $e) {
            \Log::error('Image optimization failed: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Delete optimized image variants
     * 
     * @param string $imagePath
     * @return bool
     */
    public static function deleteImageVariants(string $imagePath): bool
    {
        if (!file_exists($imagePath)) {
            return false;
        }

        $pathInfo = pathinfo($imagePath);
        $basePath = $pathInfo['dirname'];
        $filename = $pathInfo['filename'];

        $filesToDelete = [
            $imagePath, // Original
            "{$basePath}/{$filename}.webp",
            "{$basePath}/{$filename}-large.jpg",
            "{$basePath}/{$filename}-large.webp",
            "{$basePath}/{$filename}-medium.jpg",
            "{$basePath}/{$filename}-medium.webp",
            "{$basePath}/{$filename}-small.jpg",
            "{$basePath}/{$filename}-small.webp",
        ];

        foreach ($filesToDelete as $file) {
            if (file_exists($file)) {
                unlink($file);
            }
        }

        return true;
    }
}
