<?php

namespace Database\Seeders;

use App\Models\Room;
use App\Models\RoomType;
use App\Models\User;
use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;

class DatabaseSeeder extends Seeder
{
    use WithoutModelEvents;

    /**
     * Seed the application's database.
     */
    public function run(): void
    {
        $admin = User::updateOrCreate(
            ['email' => 'admin@hotili.test'],
            [
                'name' => 'Hotili Admin',
                'role' => User::ROLE_ADMIN,
                'password' => Hash::make('Admin@1234'),
            ]
        );

        User::updateOrCreate(
            ['email' => 'reception@hotili.test'],
            [
                'name' => 'Reception Desk',
                'role' => User::ROLE_RECEPTION,
                'password' => Hash::make('Reception@1234'),
            ]
        );

        $roomTypes = [
            [
                'name' => 'Single Room',
                'description' => 'Cozy and elegant space ideal for solo travelers with premium bedding and modern comforts.',
                'base_price' => 4500,
                'capacity' => 1,
                'area' => 22,
                'amenities' => ['Free Wi-Fi', 'Smart TV', 'Air Conditioning', 'Breakfast Included'],
            ],
            [
                'name' => 'Double Room',
                'description' => 'Spacious room for couples or friends, featuring a comfortable double bed and refined finishes.',
                'base_price' => 6500,
                'capacity' => 2,
                'area' => 28,
                'amenities' => ['Free Wi-Fi', 'Smart TV', 'Mini Fridge', 'City View'],
            ],
            [
                'name' => 'Deluxe Room',
                'description' => 'Enhanced comfort with extra space, lounge seating, and premium bath amenities.',
                'base_price' => 8500,
                'capacity' => 2,
                'area' => 34,
                'amenities' => ['Free Wi-Fi', 'Smart TV', 'Coffee Maker', 'Balcony'],
            ],
            [
                'name' => 'Executive Suite',
                'description' => 'A luxurious suite with separate living area, perfect for extended stays or VIP guests.',
                'base_price' => 12000,
                'capacity' => 3,
                'area' => 48,
                'amenities' => ['Free Wi-Fi', 'Smart TV', 'Living Area', 'Premium Toiletries'],
            ],
            [
                'name' => 'Family Suite',
                'description' => 'A spacious multi-bed suite designed for family comfort, with lounge space and kid-friendly conveniences.',
                'base_price' => 14500,
                'capacity' => 4,
                'area' => 56,
                'amenities' => ['Free Wi-Fi', 'Smart TV', 'Two Queen Beds', 'Lounge Area', 'Family Dining Nook'],
            ],
            [
                'name' => 'Presidential Suite',
                'description' => 'Our most exclusive suite with panoramic views, private lounge, and elevated in-room services.',
                'base_price' => 18500,
                'capacity' => 4,
                'area' => 72,
                'amenities' => ['Free Wi-Fi', 'Smart TV', 'Private Lounge', 'Panoramic View', 'Premium Concierge Service'],
            ],
        ];

        $createdRoomTypes = [];

        foreach ($roomTypes as $roomTypeData) {
            $createdRoomTypes[$roomTypeData['name']] = RoomType::updateOrCreate(
                ['name' => $roomTypeData['name']],
                $roomTypeData
            );
        }

        $rooms = [
            'Single Room' => ['101', '102', '103'],
            'Double Room' => ['201', '202', '203'],
            'Deluxe Room' => ['301', '302'],
            'Executive Suite' => ['401', '402'],
            'Family Suite' => ['501', '502'],
            'Presidential Suite' => ['601', '602'],
        ];

        foreach ($rooms as $typeName => $roomNumbers) {
            $roomType = $createdRoomTypes[$typeName] ?? null;
            if (!$roomType) {
                continue;
            }

            foreach ($roomNumbers as $roomNumber) {
                Room::updateOrCreate(
                    ['room_number' => $roomNumber],
                    [
                        'room_type_id' => $roomType->id,
                        'floor' => intdiv((int) $roomNumber, 100),
                        'status' => Room::STATUS_AVAILABLE,
                    ]
                );
            }
        }
    }
}
