import { Layout } from "@/components/Layout";
import { Button } from "@/components/ui/button";
import { Users, Maximize, Wifi, Coffee } from "lucide-react";
import deluxeRoom from "@/assets/room-deluxe.jpg";
import executiveRoom from "@/assets/room-executive.jpg";
import presidentialRoom from "@/assets/room-presidential.jpg";
import { useRoomTypes } from "@/hooks/useRoomTypes";

const fallbackRooms = [
  {
    image: deluxeRoom,
    name: "Deluxe Room",
    description: "Experience refined comfort in our thoughtfully designed Deluxe Room. Featuring a plush king-size bed, city views, and modern amenities for a relaxing stay.",
    price: 150,
    size: "35 sqm",
    guests: 2,
    amenities: ["King Bed", "City View", "Mini Bar", "Smart TV", "Free Wi-Fi", "Room Service"],
  },
  {
    image: executiveRoom,
    name: "Executive Suite",
    description: "Elevate your experience in our spacious Executive Suite. Enjoy a separate living area, premium furnishings, and exclusive access to the Executive Lounge.",
    price: 280,
    size: "55 sqm",
    guests: 3,
    amenities: ["King Bed", "Living Area", "City View", "Mini Bar", "Smart TV", "Executive Lounge", "Butler Service"],
  },
  {
    image: presidentialRoom,
    name: "Presidential Suite",
    description: "The pinnacle of luxury at Hotili. Panoramic views, private terrace, personal butler, and bespoke services for the most discerning guests.",
    price: 500,
    size: "120 sqm",
    guests: 4,
    amenities: ["King Bed", "Living Room", "Dining Area", "Private Terrace", "Panoramic View", "Personal Butler", "Luxury Bathroom", "Private Check-in"],
  },
  {
    image: deluxeRoom,
    name: "Superior Room",
    description: "A cozy retreat with essential luxury amenities. Perfect for solo travelers or couples seeking comfort at an exceptional value.",
    price: 120,
    size: "28 sqm",
    guests: 2,
    amenities: ["Queen Bed", "City View", "Smart TV", "Free Wi-Fi", "Room Service"],
  },
  {
    image: executiveRoom,
    name: "Family Suite",
    description: "Spacious accommodations for the whole family. Two bedrooms, a comfortable living space, and amenities designed with families in mind.",
    price: 350,
    size: "75 sqm",
    guests: 5,
    amenities: ["2 Bedrooms", "Living Area", "Kitchen", "Kids Amenities", "Smart TV", "Free Wi-Fi", "Room Service"],
  },
  {
    image: presidentialRoom,
    name: "Honeymoon Suite",
    description: "A romantic haven for newlyweds. Rose petals, champagne on arrival, private dining, and breathtaking views create unforgettable memories.",
    price: 400,
    size: "65 sqm",
    guests: 2,
    amenities: ["King Bed", "Jacuzzi", "Champagne", "Romantic Setup", "City View", "Private Dining", "Butler Service"],
  },
];

const Rooms = () => {
  const { data: roomTypes } = useRoomTypes();

  const rooms = roomTypes?.length
    ? roomTypes.map((roomType, index) => {
        const fallback =
          fallbackRooms[index % fallbackRooms.length]?.image ?? deluxeRoom;

        return {
          image: roomType.image_url || fallback,
          fallbackImage: fallback,
          name: roomType.name,
          description:
            roomType.description ||
            "A refined space designed for comfort, style, and restful nights.",
          price: Number(roomType.base_price || 0),
          size: roomType.area ? `${roomType.area} sqm` : "35 sqm",
          guests: roomType.capacity || 2,
          amenities:
            roomType.amenities?.length ? roomType.amenities : ["Free Wi-Fi", "Smart TV", "Room Service", "Air Conditioning"],
        };
      })
    : fallbackRooms.map((room) => ({ ...room, fallbackImage: room.image }));

  return (
    <Layout>
      {/* Hero Section */}
      <section className="pt-32 pb-16 bg-primary">
        <div className="container mx-auto px-4 lg:px-8 text-center">
          <span className="text-gold text-sm tracking-[0.2em] uppercase font-medium">
            Accommodations
          </span>
          <h1 className="text-4xl md:text-5xl font-serif font-bold text-primary-foreground mt-3 mb-4">
            Rooms & Suites
          </h1>
          <div className="gold-divider mx-auto" />
          <p className="text-primary-foreground/80 mt-6 max-w-2xl mx-auto">
            Discover our collection of elegantly appointed rooms and suites, each designed 
            to provide the perfect blend of comfort and luxury.
          </p>
        </div>
      </section>

      {/* Rooms Grid */}
      <section className="py-20 bg-background">
        <div className="container mx-auto px-4 lg:px-8">
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
            {rooms.map((room, index) => (
              <div
                key={index}
                className="group bg-card rounded-xl overflow-hidden shadow-card hover:shadow-elevated transition-all duration-500 gold-border"
              >
                {/* Room Image */}
                <div className="relative h-56 overflow-hidden">
                  <img
                    src={room.image}
                    alt={room.name}
                    className="w-full h-full object-cover transition-transform duration-700 group-hover:scale-110"
                    onError={(event) => {
                      event.currentTarget.src = room.fallbackImage;
                    }}
                  />
                  <div className="absolute top-4 right-4 bg-gold text-charcoal px-3 py-1 rounded-full text-sm font-semibold">
                    KSh {room.price.toLocaleString()}/night
                  </div>
                </div>

                {/* Room Info */}
                <div className="p-6">
                  <h3 className="text-xl font-serif font-semibold text-foreground mb-2">
                    {room.name}
                  </h3>
                  
                  {/* Room Meta */}
                  <div className="flex items-center gap-4 text-muted-foreground text-sm mb-4">
                    <div className="flex items-center gap-1">
                      <Maximize className="w-4 h-4 text-gold" />
                      <span>{room.size}</span>
                    </div>
                    <div className="flex items-center gap-1">
                      <Users className="w-4 h-4 text-gold" />
                      <span>{room.guests} Guests</span>
                    </div>
                  </div>

                  <p className="text-muted-foreground text-sm mb-4 leading-relaxed line-clamp-2">
                    {room.description}
                  </p>

                  {/* Amenities Preview */}
                  <div className="flex flex-wrap gap-2 mb-6">
                    {room.amenities.slice(0, 4).map((amenity, i) => (
                      <span
                        key={i}
                        className="text-xs bg-muted text-muted-foreground px-2 py-1 rounded"
                      >
                        {amenity}
                      </span>
                    ))}
                    {room.amenities.length > 4 && (
                      <span className="text-xs text-gold">
                        +{room.amenities.length - 4} more
                      </span>
                    )}
                  </div>

                  <Button variant="gold" className="w-full">
                    Book Now
                  </Button>
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="py-16 bg-primary">
        <div className="container mx-auto px-4 lg:px-8 text-center">
          <h2 className="text-2xl md:text-3xl font-serif font-bold text-primary-foreground mb-4">
            Need Help Choosing?
          </h2>
          <p className="text-primary-foreground/80 mb-8 max-w-xl mx-auto">
            Our team is here to help you find the perfect room for your stay. 
            Contact us for personalized recommendations.
          </p>
          <Button variant="goldOutline" size="lg">
            Contact Reservations
          </Button>
        </div>
      </section>
    </Layout>
  );
};

export default Rooms;
